#!/usr/bin/env python

"""
The controller sets the speed of a fan according to temperature (mode 1),
or lets the fan run with a constant speed for a given time (mode 2).
Temperature is measured with Pmod TC1.
The motor is driven by Pmod HB3.
The user interface is realized with Pmod OLEDrgb and Pmod KYPD.
"""

# import necessary modules
from DesignSpark.Pmod.HAT import createPmod
from luma.core.render import canvas
from datetime import datetime, timedelta
import time

# import own modules
import display
import keypad

# create objects for each pmod
TC1 = createPmod('TC1', 'JBA')
HB3 = createPmod('HB3', 'JBB')
keypad.KYPD = createPmod('KYPD', 'JC')  # Pmod KYPD and Pmod OLEDrgb are
display.OLED = createPmod('OLEDrgb', 'JA')  # used in separate modules


class settings:
    """ this class contains "global variables" """
    TEMP_AVG = 500  # number of temperature measurements to average
    TEMP_MIN = 22.0  # temperature for 0% fan speed
    TEMP_MAX = 35.0  # temperature for 100% fan speed
    TIME_RUN = 15  # number of minutes to run (mode 2)
    CONST_SPEED = 75    # speed in mode 2
    MOTOR_REV = False  # reverse the rotation of the motor
    MODE = 1  # starting mode


def action_M1():
    """ this is done in mode 1 """
    # get average temperature
    average_temp = 0.0
    for _ in range(settings.TEMP_AVG):
        average_temp = average_temp + TC1.readCelcius()
    average_temp = average_temp / settings.TEMP_AVG

    # calculate the speed from the temperature
    average_temp = average_temp - settings.TEMP_MIN
    max_temp = settings.TEMP_MAX - settings.TEMP_MIN
    if max_temp <= 0:
        max_temp = 1
    speed = 0
    if average_temp > 0:
        speed = average_temp * 100 / max_temp

    # limit the speed
    if speed > 100:
        speed = 100
    elif speed < 0:
        speed = 0

    # set the speed of the motor
    if settings.MOTOR_REV:
        HB3.reverse(speed)
    else:
        HB3.forward(speed)

    # display data
    display.data_M1(average_temp, speed, settings.TEMP_MIN,
                    settings.MODE)
    return


def action_M2():
    """ this is done in mode 2 """
    # calculate stop time
    stop_time = datetime.now() + timedelta(minutes=settings.TIME_RUN)

    # set motor speed
    if settings.MOTOR_REV:
        HB3.reverse(settings.CONST_SPEED)
    else:
        HB3.forward(settings.CONST_SPEED)

    # wait but check buttons
    try:
        # if the time didn't expire, continue
        while datetime.now() < stop_time:
            # display data in every iteration (the time will change)
            display.data_M2(settings.CONST_SPEED, stop_time -
                            datetime.now(), settings.MODE)
            # check for keypress
            if keypad.KYPD.getKey() != None:
                raise keypad.KeyPressed
            pass
    # on keypress, stop the motor and exit
    except keypad.KeyPressed:
        HB3.forward(0)
        return

    # calculate stop time
    stop_time = datetime.now() + timedelta(minutes=settings.TIME_RUN)

    # stop motor
    HB3.reverse(0)

    # wait but check buttons
    try:
        # if the time didn't expire, continue
        while datetime.now() < stop_time:
            # display data in every iteration (the time will change)
            display.data_M2(0, stop_time - datetime.now(), settings.MODE)
            # check for keypress
            if keypad.KYPD.getKey() != None:
                raise keypad.KeyPressed
            pass
    # on keypress, stop the motor and exit
    except keypad.KeyPressed:
        HB3.forward(0)
        return
    return


def set_param(text, unit):
    """ get a parameter and display it """
    # display a text to ask for the parameter
    display.require(text)
    # read the parameter
    parameter = keypad.get_param()
    # display the parameter
    display.parameter(text, parameter, unit)
    # wait for a button to exit
    keypad.wait_for_key()
    return parameter


def decode_key(stage, key=None):
    """ this is done, when a key is pressed """
    # skip the main menu, if the key was pressed,
    # when a submenu was displayed
    if stage == 0:
        # display the main menu
        display.menu()
        # wait for a keypress
        key = keypad.wait_for_key()

    # if "A" is pressed in the main menu, or mode 1 submenu
    # was requested
    if key == "A" or stage == 1:
        # set mode to 1
        settings.MODE = 1
        # display submenu for mode 1
        display.menu_M1()
        # wait for a keypress
        key = keypad.wait_for_key()
        # if "C" is pressed in the submenu
        if key == "C":
            # request and set the minimum temperature
            settings.TEMP_MIN = set_param("low temperature limit", "\xb0C")
            # go back to the mode 1 submenu
            decode_key(1)
        # if "D" is pressed in the submenu
        elif key == "D":
            # request and set the minimum temperature
            settings.TEMP_MAX = set_param("high temperature limit", "\xb0C")
            # go back to the mode 1 submenu
            decode_key(1)
        else:
            # go back to the main menu
            decode_key(0)

    # if "B" is pressed in the main menu, or mode 2 submenu
    # was requested
    elif key == "B" or stage == 2:
        # set mode to 2
        settings.MODE = 2
        # display submenu for mode 2
        display.menu_M2()
        # wait for keypress
        key = keypad.wait_for_key()
        # if "C" is pressed in the submenu
        if key == "C":
            # request and set the speed
            settings.CONST_SPEED = set_param("speed", "%")
            # go back to the mode 2 submenu
            decode_key(2)
        # if "D" is pressed in the submenu
        elif key == "D":
            # request and set the runtime
            settings.TIME_RUN = set_param("nr. of minutes", "min")
            # go back to the mode 2 submenu
            decode_key(2)
        else:
            # go back to the main menu
            decode_key(0)
    return


# main program
try:
    # setup
    keypad.KYPD.setKeyMapDefault()  # set default key map
    time.sleep(1)  # wait a second

    # main loop
    while True:
        # if no key was pressed
        if keypad.get_key() == None:
            # if the mode variable is 1
            if settings.MODE == 1:
                # call function for mode 1
                action_M1()
            # if the mode variable is 2
            elif settings.MODE == 2:
                # call function for mode 2
                action_M2()
        # if a key was pressed
        else:
            # stop the motor
            HB3.forward(0)
            # decode the key
            decode_key(0)

except KeyboardInterrupt:
    # exit on ctrl+c
    pass
finally:
    # close opened connections
    TC1.cleanup()
    HB3.cleanup()
    keypad.KYPD.cleanup()
    display.OLED.cleanup()
